import {
  getPolicy,
  getAllPolicy,
  LOGCOLOR,
  setPolicyStatus,
  setPolicy,
  setPolicyLinks,
  isRegularURL,
  showPrivacyOverlay,
  getViolations,
  getUserPreferences,
  logAction,
} from './common.js';
import {
  API_ENDPOINT,
  STORAGE_KEYS,
  PREFERENCE_LEVELS,
  LOGEVENTS,
  MESSAGE_TYPES,
  SLIDER_COLORS,
  ICON_OVERLAY_COLORS,
} from './constants.js';
import { getUUID } from './user.js';

/**
 * Collect all links from the webpage.
 *
 * @param {Object} details - The details of the tab.
 * @param {string} uuid - The UUID of the user.
 */
function collectAllLinksAndSendToServer(details, uuid) {
  // Set uuid for the current user
  if (details.frameId === 0) { // Ensure it's the main frame
    chrome.scripting.executeScript({
      target: { tabId: details.tabId },
      function: () => {
        // Collect links from webpage
        const links = Array.from(document.querySelectorAll("a"))
          .map(link => ({
            text: link.textContent.trim(),
            href: link.href,
          }))
          .filter(link => link.href); // Ensure href exists

        return {
          links: links,
          baseUrl: window.location.origin,
        };
      }
    }, async (results) => {
      if (!results || !results[0] || !results[0].result) {
        // TODO Error handling
        return;
      }
      const { baseUrl, links } = results[0].result;
      setPolicyLinks(baseUrl, links)
        .then(() => updateBadge(activeTab));
      getPolicyFromServer(baseUrl, uuid, links);
    });
  }
}

/**
 * Sends a list of links for a specific base URL to the server to check if a policy is now available.
 *
 * @param {string} baseUrl - The base origin of the site.
 * @param {string} uuid - The user/device/session identifier.
 * @param {Array<string>} links - List of discovered links for that site.
 * @returns {Promise<void>}
 */
function getPolicyFromServer(baseUrl, uuid, links) {
  console.info('%cSending links:', LOGCOLOR('#4CAF50', 'white'), links);

  fetch(API_ENDPOINT + "policies", {
    method: "POST",
    headers: {
      "Content-Type": "application/json"
    },
    body: JSON.stringify({ uuid: uuid, url: baseUrl, links: links })
  }).then(response => {
    if (response.status === 200) {
      return response.json();
    } else if (response.status === 202) {
      console.info(`Policy for ${baseUrl} is still processing.`);
      return null;
    }
  }).then((policy) => {
    if (policy) {
      if (policy.error) {
        console.error(`Error sending policy for ${baseUrl}:`, policy.error);
        setPolicyStatus(baseUrl, "resend");
        return;
      } else {
        const {baseurl, p3p, policy_url} = policy;
        console.info(`Policy for ${baseUrl} done:`, policy);
        setPolicy(baseUrl, p3p)
          .then(() => updateBadge(activeTab));
      }
    }
  }).catch((error) => {
    // This is a fallback, if the server is not reachable or no internet connection is available
    setPolicyStatus(baseUrl, "resend");
  });
}

/**
 * Changing the badge to a checkmark.
 *
 * @param {number} tabId - The ID of the tab to show the checkmark.
 */
function showSuccessBadge(tabId) {
  chrome.action.setBadgeText({ tabId, text: "✓" });
  chrome.action.setBadgeBackgroundColor({ tabId, color: ICON_OVERLAY_COLORS.SUCCESS });
  chrome.action.setBadgeTextColor({ tabId, color: "white" });
}

/**
 * Changing the badge to a unknown state.
 *
 * @param {number} tabId - The ID of the tab to show the unknown state.
 */
function showUnknownState(tabId) {
  chrome.action.setBadgeText({ tabId, text: "?" });
  chrome.action.setBadgeBackgroundColor({ tabId, color: ICON_OVERLAY_COLORS.UNKNOWN });
}

/**
 * Changing the badge to a danger state.
 *
 * @param {number} tabId - The ID of the tab to show the danger state.
 */
function showDangerBadge(tabId) {
  chrome.action.setBadgeText({ tabId, text: "!" });
  chrome.action.setBadgeBackgroundColor({ tabId, color: ICON_OVERLAY_COLORS.BLOCK });
  chrome.action.setBadgeTextColor({ tabId, color: "white" });
}

/**
 * Changing the badge to a danger state.
 *
 * @param {number} tabId - The ID of the tab to show the danger state.
 */
function showWarningBadge(tabId) {
  chrome.action.setBadgeText({ tabId, text: "!" });
  chrome.action.setBadgeBackgroundColor({ tabId, color: ICON_OVERLAY_COLORS.WARN });
  chrome.action.setBadgeTextColor({ tabId, color: "black" });
}

/**
 * Changing the badge to a clear state.
 *
 * @param {number} tabId - The ID of the tab to clear the badge.
 */
function showClearState(tabId) {
  chrome.action.setBadgeText({ tabId, text: "" });
  chrome.action.setBadgeBackgroundColor({ tabId, color: "#00AA00" });
}


function isTabActive (tabId) {
  return new Promise((resolve, reject) => {
    chrome.tabs.get(tabId, function(tab) {
      if (chrome.runtime.lastError) {
        // Happens if tabId is invalid or closed
        resolve(false);
        return;
      }
      resolve(Boolean(tab.active));
    });
  });
}


// mark tabs that just reloaded
chrome.webNavigation.onCommitted.addListener(
  (details) => {
    if (details.frameId !== 0) return;
    if (details.transitionType === "reload") {
      return
    }
  },
  { url: [{ schemes: ["http", "https"] }] }
);


/**
 * Checks the privacy policy of a given URL for the specified tab and determines
 * if it should be blocked based on user preferences, whitelist, or temporary
 * bypasses. If a violation is detected, displays a privacy overlay.
 *
 * @async
 * @function checkPolicyForBlocks
 * @param {string} url - The URL to check the privacy policy for.
 * @param {number} tabId - The ID of the browser tab to apply the check to.
 * @returns {Promise<void>} Resolves when the check is complete.
 */
async function checkPolicyForBlocks (url, tabId) {
  const whitelistedSites = await chrome.storage.local.get(STORAGE_KEYS.WHITELIST);
  const whitelist = whitelistedSites[STORAGE_KEYS.WHITELIST] || { };

  const temporaryBypasses = await chrome.storage.local.get(STORAGE_KEYS.BYPASS);
  const bypasses = temporaryBypasses[STORAGE_KEYS.BYPASS] || { };

  url = new URL(url).origin;

  if (url in whitelist) {
    // If the site is whitelisted, we can allow it
    console.info('Webpage whitelisted');
  } else if (url in bypasses && bypasses[url].includes(tabId)) {
    // If the site is temporarily bypassed, we can allow it
    console.info('Webpage temporarily bypassed');
  } else {
    getPolicy(url).then(async (policyInfo) => {
      if (!policyInfo || policyInfo.status !== "done") {
        return;
      } else if (policyInfo.status === "done") {
        const  { policy } = policyInfo;

        // Check if policy violates user preferences
        const preferences = await getUserPreferences();
        const violations = getViolations(policy, preferences);

        // if any of the policies includes block
        if (violations.some(violation => violation.level === Object.values(PREFERENCE_LEVELS).indexOf(PREFERENCE_LEVELS.BLOCK))) {
          console.warn('%cPolicy violates user preferences', LOGCOLOR('red', 'white'), url, violations);
          showPrivacyOverlay(tabId, `${url} may collect data in ways that conflict with your preferences.`);
        }
      }
    });
  }
}


/**
 * On closing the current tab, remove it from any bypass entries
 */
chrome.tabs.onRemoved.addListener((tabId, removeInfo) => {
  chrome.storage.local.get(STORAGE_KEYS.BYPASS).then(({ [STORAGE_KEYS.BYPASS]: bypasses }) => {
    if (!bypasses) {
      return;
    }

    chrome.storage.local.set({ [STORAGE_KEYS.BYPASS]: Object.fromEntries(
      Object.entries(bypasses)
        .filter(([url, tabs]) => tabs.length > 0)
        .map(([url, tabs]) => [url, tabs.filter(tab => tab !== tabId)])
    )});
  });
});


/**
 * Handle the opening of a new page.
 * @param {*} details
 * @returns
 */
async function handleOpenPage(details) {
  const tabId = details.tabId;
  const url = details.url;
  console.warn(details);
  const baseUrl = new URL(url).origin;
  logAction(LOGEVENTS.PAGEVISIT, 'Opened page', { url: baseUrl });
  checkPolicyForBlocks(baseUrl, tabId);

  if (!isRegularURL(url)) {
    showClearState(tabId);
    return;
  }

  // TODO: with  `frameType :  "outermost_frame"` can we handle iframed pages?
  // TODO: what if the first page that is ever loaded had not a privacy policy (e.g., a PDF)?

  const uuid = await getUUID();

  // Check if the URL is already in the storage
  const policy = await getPolicy(baseUrl);
  // console.log(url, policy);
  if (!policy && details.statusCode === 200) {
    console.info(`No policy found for ${baseUrl}, sending processing request to server...`);
    collectAllLinksAndSendToServer(details, uuid);
  } else if (policy && policy.status === "resend") {
    console.info(`Policy for ${baseUrl} needs to be resent, fetching links...`);
    // Collect all links and send them to the server
    collectAllLinksAndSendToServer(details, uuid);
    // getPolicyFromServer(baseUrl, uuid, policy.links);
  } else if (policy && policy.status === "processing") {
    console.info(`Policy for ${baseUrl} is still processing.`);
    // Collect all links and send them to the server
    collectAllLinksAndSendToServer(details, uuid);
    // getPolicyFromServer(baseUrl, uuid, policy.links);
  } else if (policy && policy.status === "done") {
    // console.info(`Policy for ${baseUrl} is already processed.`);
  }

  updateBadge(tabId, baseUrl);

}

/**
 * Listener function which gets called when the user opens a new page.
 *
 * @listens chrome.webRequest.onCompleted
 * @param {Object} details - The details of the request.
 * @returns {Promise<void>}
 */
chrome.webRequest.onCompleted.addListener(
  async (details) => {
    console.info('WebRequest completed:', details);
    const { tabId, type } = details;
    if (type !== "main_frame" || tabId === -1) {
      return;
    }
    handleOpenPage(details);
    if (await isTabActive(tabId)) {
      activeTab = await chrome.tabs.get(tabId);
      console.info('%cActiveTab', LOGCOLOR('orange', 'white'), activeTab);
      updateBadge(tabId, details.url);
    }
    openPopupIfNewPageVisit(tabId, details.url);
  },
  { urls: ["<all_urls>"], types: ["main_frame"] }
);


function openPopupIfNewPageVisit (tabId, url) {
  const baseUrl = new URL(url).origin;
  chrome.storage.local.get(STORAGE_KEYS.VISITED_URLS_IN_TAB).then(({ [STORAGE_KEYS.VISITED_URLS_IN_TAB]: visitedUrlsInTab }) => {
    visitedUrlsInTab = visitedUrlsInTab || { };
    const lastVisited = visitedUrlsInTab[baseUrl];

    if (lastVisited && (new Date() - new Date(lastVisited)) < (2 * 24 * 60 * 60 * 1000)) {
      // If the last visit was less than 2 days ago, do nothing
      return;
    } else if (!lastVisited) {
      // If this is the first visit, we set the last visited time
      visitedUrlsInTab[baseUrl] = new Date().toISOString();
    }

    getPolicy(baseUrl).then((policy) => {
      if (policy && policy.status === "done") {
        const preferences = getUserPreferences();
        const violations = getViolations(policy.policy, preferences);

        const r = Math.random();
        console.log('%cRandom number for popup decision:', LOGCOLOR('black', 'white'), r, violations.length);

        visitedUrlsInTab[baseUrl] = new Date().toISOString();
        chrome.storage.local.set({ [STORAGE_KEYS.VISITED_URLS_IN_TAB]: visitedUrlsInTab });
        logAction(LOGEVENTS.OPEN_POPUP, 'Opened popup for new page visit (no violations)', { url: baseUrl });
        chrome.action.openPopup();
      }
    });
  });
}


/**
 * Listener function which gets called when the navigation changes; e.g., a new page is loaded.
 */
chrome.webNavigation.onCompleted.addListener(async (details) => {
  const tabId = details.tabId;

  if (!isRegularURL(details.url)) {
    updateBadge(tabId, details.url);
    return;
  }
  handleOpenPage(details);
  if (await isTabActive(tabId)) {
    activeTab = await chrome.tabs.get(tabId);
    console.info('%cActiveTab', LOGCOLOR('orange', 'white'), activeTab);
    updateBadge(tabId, details.url);
  }
  openPopupIfNewPageVisit(tabId, details.url);
});

let activeTab = null;

/**
 * Listener function which gets called when the user switches to a different tab.
 *
 * @listens chrome.tabs.onActivated
 * @param {Object} activeInfo - The active tab information.
 */
chrome.tabs.onActivated.addListener(async function(activeInfo) {
  const tab = await chrome.tabs.get(activeInfo.tabId);

  activeTab = tab;
  console.info('%cActiveTab', LOGCOLOR('orange', 'white'), activeTab);

  if (tab.status !== 'complete') {
    return;
  }
  checkPolicyForBlocks(tab.url, tab.id);
  updateBadge(tab.id, tab.url);
});

/**
 * This is useful to update the badge when the user switches tabs.
 *
 * @param {number} tabId - The ID of the updated tab.
 * @param {string} url - The URL of the updated tab.
 */
async function updateBadge(tabId, url) {
  console.info('%cUpdating badge...', LOGCOLOR('violet', 'white'), tabId, url);
  console.trace();
  if (tabId === undefined || tabId === null || tabId < 0) {
    console.error("Invalid tabId:", tabId);
    return;
  }

  if (!url) {
    url = await chrome.tabs.get(tabId).then((tab) => tab?.url);
  }

  if (!isRegularURL(url)) {
    showClearState(tabId);
    return;
  }

  const policy = await getPolicy(url);
  try {
    if (policy && policy.status === "done") {
      // TODO: check if the page is in line with the users will via showDangerState
      const preferences = await getUserPreferences();
      const violations = getViolations(policy.policy, preferences);
      console.info(preferences, violations, tabId);

      if (violations.some(violation => violation.level === Object.values(PREFERENCE_LEVELS).indexOf(PREFERENCE_LEVELS.BLOCK))) {
        showDangerBadge(tabId);
      } else if (violations.length > 0) {
        showWarningBadge(tabId);
      } else {
        showSuccessBadge(tabId);
      }
    } else if (policy && policy.status === "processing") {
      showUnknownState(tabId);
    } else {
      showClearState(tabId);
    }
  } catch (error) {
    logAction(LOGEVENTS.ERROR, 'Error updating badge', { url, tabId, error });
    console.error(error);
    showClearState(tabId);
  }
}

/**
 * Check if P3P exists in the headers of the webpage.
 * @param {string} url - The URL of the webpage to check.
 * @param {string} uuid - The UUID of the user.
 * @returns {Promise<string|null>} - The P3P header value or null if not found.
 */
async function checkIfP3PExistsInHeaders(url, uuid) {
  // Check if P3P header exists
  let response = null;
  try {
     response = await fetch(url, { method: 'HEAD' });
  } catch (error) {
    return null;
  }
  const p3pHeader = response.headers.get('P3P');
  if (p3pHeader) {
    console.info('P3P header found:', p3pHeader);
    return p3pHeader
  } else {
    return null;
  }
}

chrome.runtime.onInstalled.addListener(() => {
  chrome.alarms.create('p4p-poll-unfinished-policies', {
    periodInMinutes: 1, // every minute
  });
  logAction(LOGEVENTS.INSTALL, 'Extension installed');

  // Also run once immediately
  getWellKnownDataTypes();
  pollUnfinishedPolicies();
});

chrome.alarms.onAlarm.addListener((alarm) => {
  getWellKnownDataTypes();
  if (alarm.name === 'p4p-poll-unfinished-policies') {
    pollUnfinishedPolicies();
  }
});

/**
 * Polls all stored policies and resends links for those whose status is not "done".
 * For each incomplete policy, calls sendLinksToServer with the stored links and UUID.
 *
 * Intended to be called on an interval or via chrome.alarms.
 *
 * @returns {Promise<void>}
 */
async function pollUnfinishedPolicies() {
  console.info("Polling unfinished policies...");
  try {
    const policies = await getAllPolicy();
    const uuid = await getUUID();

    for (const [baseUrl, entry] of Object.entries(policies)) {
      if (entry.status !== "done") {
        console.info(`Retrying policy check for: ${baseUrl}`);
        getPolicyFromServer(baseUrl, uuid, entry.links);
      }
    }
  } catch (error) {
    logAction(LOGEVENTS.ERROR, 'Error polling unfinished policies', { error });
    console.error("Error in pollUnfinishedPolicies:", error);
  }
}

/**
 * Pull well known data types from server API_ENDPOINT + policies/data-types
 */
async function getWellKnownDataTypes() {
  console.log("Fetching well known data types...");
  try {
    const uuid = await getUUID();
    fetch(`${API_ENDPOINT}policies/data-types?uuid=${uuid}&hl=en`).then(response => {
      if (!response.ok) {
        throw new Error("Network response was not ok");
      }
      return response.json();
    }).then(dataTypes => {
      console.log("Well known data types:", dataTypes);
      chrome.storage.local.set({ "wellKnownDataTypes": dataTypes });
    });
  } catch (error) {
    logAction(LOGEVENTS.ERROR, 'Error fetching well known data types', { error });
    console.error("Error fetching well known data types:", error);
  }
}


chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg?.type === MESSAGE_TYPES.CLOSE_TAB) {
    logAction(LOGEVENTS.LEAVE_PAGE, 'Tab closed by user', { url: sender.tab?.url });
    const tabId = sender.tab?.id;
    if (typeof tabId === 'number') {
      chrome.tabs.remove(tabId);
    } else {
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]?.id) chrome.tabs.remove(tabs[0].id);
      });
    }
  }

  if (msg?.type === MESSAGE_TYPES.ADD_EXCEPTION) {
    logAction(LOGEVENTS.ADD_WHITELIST, 'Added exception', { url: msg.url });
    chrome.storage.local.get(STORAGE_KEYS.WHITELIST)
      .then(({ [STORAGE_KEYS.WHITELIST]: whitelist }) => {
        whitelist = whitelist || { };
        const url = msg.url;
        // If the URL is not already whitelisted, add it
        if (!(url in whitelist)) {
          whitelist[url] = new Date().toISOString();
          chrome.storage.local.set({ [STORAGE_KEYS.WHITELIST]: whitelist })
            .then(() => chrome.runtime.sendMessage({ type: MESSAGE_TYPES.REFRESH_SIDEPANEL }));
        }
      });
  }

  if (msg?.type === MESSAGE_TYPES.SHOW_SETTINGS) {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs.length > 0) {
        const tab = tabs[0];
        logAction(LOGEVENTS.OPEN_SIDEPANEL, 'Sidepanel opened from notice', { url: tab.url });
        chrome.sidePanel.open({ windowId: tab.windowId });
      }
    });
  }

  if (msg?.type === MESSAGE_TYPES.LOG) {
    logAction(msg.action, msg.message, { url: sender.tab?.url, msg });
  }

  if (msg?.type === MESSAGE_TYPES.PREFERENCES_CHANGED) {
    updateBadge(activeTab?.id, activeTab?.url);
  }

  if (msg?.type === MESSAGE_TYPES.RECEIVED_REMOTE_DATA) {
    updateBadge(activeTab?.id, activeTab?.url);
  }

  if (msg?.type === MESSAGE_TYPES.IGNORE_WARNING) {
    addTemporaryBypass(sender.tab?.url, sender.tab?.id);
  }

});


async function addTemporaryBypass (url, tabId) {
  url = new URL(url).origin;
  const temporaryBypasses = await chrome.storage.local.get(STORAGE_KEYS.BYPASS);
  const bypasses = temporaryBypasses[STORAGE_KEYS.BYPASS] || { };

  if (!(url in bypasses)) {
    bypasses[url] = [ tabId ];
  } else {
    bypasses[url].push(tabId);
  }
  await chrome.storage.local.set({ [STORAGE_KEYS.BYPASS]: bypasses });
}


function getBroadcastMessages(uuid) {
  return fetch(`${API_ENDPOINT}messages?uuid=${uuid}`)
    .then(response => {
      if (!response.ok) {
        throw new Error("Network response was not ok");
      }
      return response.json();
    });
}


async function pollForBroadcastMessages() {
  const uuid = await getUUID();

  chrome.storage.local.get(STORAGE_KEYS.BROADCAST_MESSAGES).then(async ({ [STORAGE_KEYS.BROADCAST_MESSAGES]: broadcastMessages }) => {
    broadcastMessages = broadcastMessages || { };
    const messages = await getBroadcastMessages(uuid);
    const newMessages = messages.filter(({ id }) => !(id in broadcastMessages));

    newMessages.forEach((msg) => broadcastMessages[msg.id] = msg);

    if (newMessages.length > 0) {
      logAction(LOGEVENTS.RECEIVED_SYSTEM_BROADCAST, 'Received broadcast messages', { messages: newMessages });
    }

    chrome.storage.local.set({ [STORAGE_KEYS.BROADCAST_MESSAGES]: broadcastMessages });
  });
}

chrome.runtime.onInstalled.addListener(() => {
  chrome.alarms.create('p4p:poll-broadcast-messages', {
    periodInMinutes: .1, // every 6 seconds
  });
});

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'p4p:poll-broadcast-messages') {
    pollForBroadcastMessages();
  }
});
