import {
  LOGEVENTS,
  MESSAGE_TYPES,
  PREFERENCE_LEVELS,
  STORAGE_KEYS,
} from './constants.js';
import { getStatements, METADATA_TYPES } from './p3p.js';
import {
  DATA_TYPES_BASE,
  PURPOSE,
  RECIPIENTS,
  RETENTION,
} from './schema.js'
import { getUUID } from './user.js';


export const LOGCOLOR = (background, color) => `display: inline-block; background-color: ${background}; color: ${color || 'inherit'}; margin-right: .5rem; padding: .25rem;`;


/**
 * Checks if a URL is valid (i.e., it uses the http or https protocol).
 *
 * @param {string} url - The URL to check.
 * @returns {boolean} True if the URL is valid, false otherwise.
 */
export function isRegularURL(url) {
  // check if the url is http or https
  try {
    const parsedUrl = new URL(url);
    return (parsedUrl.protocol === 'http:' || parsedUrl.protocol === 'https:');
  } catch (error) {
    logAction(LOGEVENTS.ERROR, 'Error parsing URL', { url, error });
    console.error('Error parsing URL:', error);
    return false;
  }
}

/**
 * Retrieves the privacy policy for a given URL.
 *
 * @param {string} url - The URL to retrieve the policy for.
 * @returns {Promise<Object|null>} The policy object, or null if not found.
 */
export async function getPolicy(url){
  const baseUrl = new URL(url).origin;
  const policies = await chrome.storage.local.get(STORAGE_KEYS.POLICIES).then(({ [STORAGE_KEYS.POLICIES]: policies }) => policies || {});

  if (!policies || !policies[baseUrl]) {
    return null;
  }

  const policy = policies[baseUrl];

  if (policy) {
    // console.info(`getPolicy: Policy for ${baseUrl} found with status: ${policy.status}`);
    // If the policy is older than 2 days remove the local policy
    if (policy.last_updated && (new Date() - new Date(policy.last_updated)) > (2 * 24 * 60 * 60 * 1000)) {
      console.info(`getPolicy: Policy for ${baseUrl} is older than 2 days, removing it.`);
      delete policies[baseUrl];
      await chrome.storage.local.set({ policies: policies });
      return null;
    }
    return policy;
  } else {
    // console.warn(`getPolicy: No policy found for ${baseUrl}`);
    return null;
  }
};

/**
 * Retrieves all stored policies from chrome.storage.local.
 *
 * @returns {Promise<Object>} An object mapping base URLs to their policy entries.
 *                            Each entry should contain { status, links, policy }.
 */
export async function getAllPolicy() {
  const result = await chrome.storage.local.get('policies');
  const policies = result.policies;

  if (policies) {
    return policies;
  } else {
    return {};
  }
}

/**
 * Sets the policy links for a given URL.
 *
 * @param {string} url - The URL to set the policy links for.
 * @param {Array<Object>} newLinks - The new links to associate with the policy.
 */
export async function setPolicyLinks(url, newLinks) {
  const baseUrl = new URL(url).origin;
  const status = "processing";

  const policies = await getAllPolicy();
  if (policies[baseUrl]) {
    const links = policies[baseUrl].links;
    for (const element of newLinks) {
      if (!(links || [ ]).some(e => e.href === element.href)) {
        links.push(element);
      }
    }
    policies[baseUrl].links = links
    policies[baseUrl].status = status;
  } else {

    // const links = []
    // for (const element of newLinks) {
    //   if (!links.some(e => e.href === element.href)) {
    //     links.push(element);
    //   }
    // }

    // If no policy exists for this baseUrl, create a new one
    policies[baseUrl] = {
      status: status,
      links: links,
      policy: null,
      last_updated: null
    };
  }

  // If no policy exists for this baseUrl, create a new one
  await chrome.storage.local.set({ policies: policies }, () => {
    console.info(`setPolicyLinks for ${baseUrl}.`);
  });
}

export async function setPolicy(url, newPolicy){
  const baseUrl = new URL(url).origin;

  const policies = await getAllPolicy();

  if (policies[baseUrl]) {
    policies[baseUrl].status = "done";
    policies[baseUrl].policy = newPolicy;
    policies[baseUrl].last_updated = new Date().toISOString();
  } else {
    policies[baseUrl] = {
      status: "done",
      links: null,
      policy: newPolicy,
      last_updated: new Date().toISOString(),
    };
  }

  await chrome.storage.local.set({ policies: policies }, () => {
    console.info(`setPolicy for ${baseUrl} set to processing.`);
  });
};

export async function setPolicyStatus(url, newStatus){
  const baseUrl = new URL(url).origin;

  const policies = await getAllPolicy();
  if (policies[baseUrl]) {
    policies[baseUrl].status = newStatus;
  } else {
    policies[baseUrl] = {
      status: newStatus,
      links: null,
      policy: null,
      last_updated: null
    };
  }

  chrome.runtime.sendMessage({
    type: MESSAGE_TYPES.RECEIVED_REMOTE_DATA,
    data: {
      url: baseUrl,
      links: policies[baseUrl].links
    }
  });
  await chrome.storage.local.set({ policies: policies }, () => {
    console.info(`setPolicyStatus for ${baseUrl} to ${newStatus}.`);
  });
};

export async function logAction(key, description, values) {
  const uuid = await getUUID();

  if (!uuid) {
    console.warn(`Skipping logAction(${key}) because UUID could not be fetched.`);
    return;
  }

  const logEntry = {
    uuid,
    timestamp: new Date().toISOString(),
    key,
    description,
    values
  };

  fetch('https://p4p.medien.ifi.lmu.de/api/log', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    },
    body: JSON.stringify(logEntry)
  }).then(response => {
      if (response.ok) {``
        return response.json();
      } else {
        return null;
      }
    }).then(data => {
      if (data) {
        console.info(`✅ Action logged`, logEntry, data);
      } else {
        // console.warn("No response data for action");
      }
    }).catch(error => {
      console.error(`🚨 Network error while logging action: ${key}`);
      console.error(error);
    });
}


/**
 * Attaches click event listeners to all elements with the 'open-settings' class.
 * When clicked, opens the Chrome side panel for the current active tab and closes the current window.
 *
 * @function
 */
export function attachOpenSettingsListener() {
  document.querySelectorAll('.open-settings').forEach(link => {
    link.addEventListener('click', () => {
      console.log('Opening side panel...');
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs.length > 0) {
          const tab = tabs[0];
          chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.OPEN_SIDEPANEL, message: 'Sidepanel opened', data: { url: tab.url } });
          chrome.sidePanel.open({ windowId: tab.windowId });
        }
      });
      window.close();
    });
  });
}


/**
 * Sets the text content of all elements with the class 'current-url' to the origin of the provided URL.
 *
 * @param {string} url - The URL whose origin will be displayed in the elements.
 */
export function setURLText(url) {
  document.querySelectorAll('.current-url').forEach(element => {
    element.textContent = new URL(url).origin;
  });
}


/**
 * Sets the text content of all elements with the class 'uuid' to the provided UUID.
 * If no UUID is provided, sets the text to "No UUID available".
 *
 * @param {string} uuid - The UUID string to display. If falsy, "No UUID available" is shown.
 */
export function setUUIDText(uuid) {
  console.info('%cUUID:', LOGCOLOR('orange'), uuid);
  document.querySelectorAll('.uuid').forEach(element => {
    element.textContent = uuid || "No UUID available";
  });
}


/**
 * Fetches the human-readable name for a well-known data type.
 *
 * @param {string} dataType - The well-known data type to look up.
 * @returns {Promise<string|null>} A promise that resolves to the human-readable name or null if not found.
 */
export async function getHumanReadableDataType(dataType) {
  dataType = dataType.replace(/^#/, '');

  const values = await chrome.storage.local.get('wellKnownDataTypes').then(({wellKnownDataTypes}) => {
    if (!wellKnownDataTypes || !wellKnownDataTypes[dataType]) {
      return null;
    }
    return wellKnownDataTypes[dataType];
  })

  if (values) {
    return values;
  }

  if (dataType in DATA_TYPES_BASE) {
    return DATA_TYPES_BASE[dataType];
  }

  return { name: dataType, description: dataType };
}


/**
 * Gets a human-readable name for a metadata field.
 * @param {string} name - The name of the metadata field.
 * @param {string} type - The type of the metadata field.
 * @returns {Object|null} An object containing the name and description, or null if not found.
 */
export function getHumanReadableName (name, type) {
  name = name.replace(/^ns0:/, '');
  name = name.replace(/^#/, '');

  switch (type) {
    case METADATA_TYPES.RECIPIENT:
      return RECIPIENTS[name] || { name: name, description: name };
    case METADATA_TYPES.PURPOSE:
      return PURPOSE[name] || { name: name, description: name };
    case METADATA_TYPES.RETENTION:
      return RETENTION[name] || { name: name, description: name };
    default:
      return null;
  }
}


/**
 * Ensures the overlay script is injected into the specified tab.
 * @param {number} tabId - The ID of the tab to inject the script into.
 */
async function ensureOverlayScript(tabId) {
  await chrome.scripting.executeScript({
    target: { tabId, allFrames: false },
    files: ["./js/overlay.js"]
  });
}


/**
 * Shows a privacy overlay in the specified tab.
 * @param {number} tabId - The ID of the tab to show the overlay in.
 * @param {string} msg - The message to display in the overlay.
 */
export async function showPrivacyOverlay(tabId, msg = "This page may collect data in ways that conflict with your preferences.") {
  console.log('%cshowPrivacyOverlay', LOGCOLOR('teal', 'white'), tabId, msg);
  await ensureOverlayScript(tabId);
  chrome.tabs.sendMessage(tabId, { type: MESSAGE_TYPES.SHOW_OVERLAY, message: msg });
}


/**
 * Hides the privacy overlay in the specified tab.
 * @param {number} tabId - The ID of the tab to hide the overlay in.
 */
export function hidePrivacyOverlay(tabId) {
  chrome.tabs.sendMessage(tabId, { type: MESSAGE_TYPES.HIDE_OVERLAY });
}


/**
 * Retrieves user preferences from Chrome's local storage.
 *
 * @async
 * @function getUserPreferences
 * @returns {Promise<Object>} A promise that resolves to the user preferences object.
 */
export async function getUserPreferences () {
  return new Promise((resolve) => {
    chrome.storage.local.get(STORAGE_KEYS.PREFERENCES, (data) => {
      resolve(data.preferences || {});
    });
  });
}


/**
 * Returns an array of violations based on the given policy and user preferences.
 *
 * Iterates through each statement in the policy, and for each data reference,
 * checks if it matches a preference rule (either exact or by prefix).
 * If the preference level is not "ALLOW", a violation object is returned.
 *
 * @param {Object} policy - The policy object containing statements and data references.
 * @param {Object} preferences - An object mapping preference rules to their levels.
 * @returns {Array<Object>} Array of violation objects with the following properties:
 *   - {string} data: The data reference that violates the preference.
 *   - {string} violates: The rule or prefix that was violated.
 *   - {number} level: The preference level associated with the violation.
 */
export function getViolations (policy, preferences) {
  return getStatements(policy).flatMap((statement) => {
    return statement.data.map((data) => {
      const rule = data.ref.replace('#', '');
      if (rule in preferences) {
        // Exact match
        if (preferences[rule] !== Object.values(PREFERENCE_LEVELS).indexOf(PREFERENCE_LEVELS.ALLOW)) {
          return { data: data.ref, violates: rule, level: preferences[rule] };
        }
      } else {
        // Attempt prefix match
        const prefixMatch = Object.keys(preferences).find((key) => rule.startsWith(key));
        if (prefixMatch && preferences[prefixMatch] !== Object.values(PREFERENCE_LEVELS).indexOf(PREFERENCE_LEVELS.ALLOW)) {
          return { data: data.ref, violates: prefixMatch, level: preferences[prefixMatch] }
        } else {
          return { data: data.ref, violates: 'unknown', level: -1 };
        }
      }
      return null;
    }).filter((violation) => violation);
  });
}