// overlay.js

(function () {
  const OVERLAY_ID = "p4p-privacy-overlay";

  const P4P_MESSAGE_TYPES = Object.freeze({
    LOG: 'p4p:log',
    CLOSE_TAB: 'p4p:closeTab',
    REFRESH_SIDEPANEL: 'p4p:refreshSidepanel',
    SHOW_SETTINGS: 'p4p:showSettings',
    ADD_EXCEPTION: 'p4p:addException',
    SHOW_OVERLAY: 'p4p:showOverlay',
    HIDE_OVERLAY: 'p4p:hideOverlay',
    PREFERENCES_CHANGED: 'p4p:preferencesChanged',
    RECEIVED_REMOTE_DATA: 'p4p:receivedRemoteData',
    IGNORE_WARNING: 'p4p:ignoreWarning',
  });

  /**
   * Creates a privacy overlay.
   * @param {string} message - The message to display in the overlay.
   * @returns {void}
   */
  function createOverlay(message = "Possible privacy issue on this page") {
    if (document.getElementById(OVERLAY_ID)) {
      return;
    }

    // Put overlay in shadow dom
    const rootElement = document.createElement('div');
    rootElement.id = OVERLAY_ID;
    const shadowRoot = rootElement.attachShadow({ mode: 'closed' });
    document.body.appendChild(rootElement);

    // Root overlay
    const overlay = document.createElement("div");
    overlay.setAttribute("role", "alert");
    overlay.setAttribute("aria-live", "assertive");
    overlay.style.position = "fixed";
    overlay.style.inset = "0";
    overlay.style.background = "rgba(40, 5, 5, 0.8)"; // transparent red
    overlay.style.zIndex = "2147483647";
    overlay.style.display = "flex";
    overlay.style.alignItems = "center";
    overlay.style.justifyContent = "center";
    overlay.style.backdropFilter = "blur(8px)";

    // Message card (clickable)
    const card = document.createElement("div");
    card.style.pointerEvents = "auto";
    card.style.background = "white";
    card.style.borderRadius = "4px";
    card.style.boxShadow = "0 6px 24px rgba(0,0,0,0.2)";
    card.style.maxWidth = "50vw";
    card.style.minWidth = "32rem";
    card.style.margin = "1rem";
    card.style.maxHeight = "80vh";
    card.style.padding = "0";
    card.style.fontFamily = "system-ui, -apple-system, Segoe UI, Roboto, sans-serif";
    card.style.color = "#111";
    card.style.lineHeight = "1.4";

    const title = document.createElement("div");
    title.textContent = "Privacy notice";
    title.style.fontWeight = "bold";
    title.style.fontSize = "18px";
    title.style.marginBottom = "6px";
    title.style.padding = '.5rem';
    title.style.borderRadius = '4px 4px 0 0';
    title.style.backgroundColor = 'rgba(185, 57, 57, 1)';
    title.style.color = 'white';

    const body = document.createElement("p");
    body.textContent = message;
    body.style.margin = '.5rem 1rem';

    const actions = document.createElement("div");
    actions.style.display = "flex";
    actions.style.gap = ".5rem";
    actions.style.padding = '0 1rem';
    actions.style.justifyContent = "flex-end";

    const btnCloseTab = document.createElement("button");
    btnCloseTab.textContent = "Leave Page";

    styleButton(btnCloseTab, "rgba(185, 57, 57, 1)", "#fff", "rgba(185, 57, 57, 1)");
    // styleButton(btnCloseTab, "#eee", "rgba(185, 57, 57, 1)");
    btnCloseTab.addEventListener("click", closeTab);

    const btnBypass = document.createElement("button");
    btnBypass.textContent = "Ignore for this tab";
    styleButton(btnBypass, "rgba(185, 57, 57, 1)", "#fff", "rgba(185, 57, 57, 1)");
    btnBypass.addEventListener("click", () => {
      chrome.runtime?.sendMessage?.({ type: P4P_MESSAGE_TYPES.LOG, action: 'ignore_warning', url: new URL(window.location.href).origin });

      chrome.runtime?.sendMessage?.({ type: P4P_MESSAGE_TYPES.IGNORE_WARNING });
      removeOverlay();
    });

    const btnException = document.createElement("button");
    btnException.textContent = "Always ignore for this page";
    styleButton(btnException, "rgba(185, 57, 57, 1)", "#fff", "rgba(185, 57, 57, 1)");
    btnException.addEventListener("click", () => {
      chrome.runtime?.sendMessage?.({ type: P4P_MESSAGE_TYPES.ADD_EXCEPTION, url: new URL(window.location.href).origin });
      removeOverlay();
    });

    const settingsBtn = document.createElement("button");
    settingsBtn.textContent = "Settings";
    styleButton(settingsBtn, "rgba(185, 57, 57, 1)", "#fff", "rgba(185, 57, 57, 1)");
    settingsBtn.addEventListener("click", () => {
      chrome.runtime?.sendMessage?.({ type: P4P_MESSAGE_TYPES.SHOW_SETTINGS });
    });

    const footer = document.createElement("footer");
    footer.style.display = "flex";
    footer.style.justifyContent = "space-around";
    footer.style.padding = ".5rem 1rem";
    footer.style.color = '#bbbbbb';
    footer.style.fontSize = '80%';
    footer.textContent = 'This warning is part of the P4P privacy extension.';

    actions.append(btnBypass, btnException, btnCloseTab, settingsBtn);
    card.append(title, body, actions, footer);
    overlay.append(card);

    shadowRoot.append(overlay);

    btnBypass.focus();
  }

  /**
   * Styles a button element.
   * @param {HTMLElement} btn - The button element to style.
   * @param {string} bg - The background color.
   * @param {string} fg - The text color.
   * @param {string} border - The border color.
   */
  function styleButton(btn, bg, fg) {
    btn.style.background = bg;
    btn.style.color = fg;
    btn.style.border = 'none';
    btn.style.borderRadius = "4px";
    btn.style.padding = "8px 12px";
    btn.style.fontSize = "14px";
    btn.style.cursor = "pointer";
  }

  /**
   * Removes the privacy overlay.
   */
  function removeOverlay() {
    const el = document.getElementById(OVERLAY_ID);
    if (el) el.remove();
  }

  /**
   * Closes the current tab.
   */
  function closeTab() {
    // Ask background to close the tab that injected this script
    chrome.runtime.sendMessage({ type: P4P_MESSAGE_TYPES.CLOSE_TAB });
    // Optionally hide your overlay immediately
    const el = document.getElementById('p4p-privacy-overlay');
    if (el) {
      el.remove();
    }
  }

  // Listen for commands from background/popup
  chrome.runtime.onMessage.addListener((msg) => {
    console.warn('receivedMessage', msg);
    if (msg?.type === P4P_MESSAGE_TYPES.SHOW_OVERLAY) {
      console.info('Showing overlay', msg.message);
      createOverlay(msg.message);
      chrome.runtime.sendMessage({ type: P4P_MESSAGE_TYPES.LOG, action: 'show_overlay', url: new URL(window.location.href).origin });
    }
    if (msg?.type === P4P_MESSAGE_TYPES.HIDE_OVERLAY) {
      console.info('Hiding overlay');
      removeOverlay();
      chrome.runtime.sendMessage({ type: P4P_MESSAGE_TYPES.LOG, action: 'hide_overlay', url: new URL(window.location.href).origin });
    }
  });

  // Expose minimal API (optional, for debugging)
  window.__P4P_OVERLAY__ = { show: createOverlay, hide: removeOverlay };
})();
