import { LOGCOLOR } from './common.js';

/**
 * Metadata types for privacy policies.
 */
export const METADATA_TYPES = {
  RECIPIENT: 'recipient',
  PURPOSE: 'purpose',
  RETENTION: 'retention',
  CONSEQUENCE: 'consequence',
};


/**
 * Recursively retrieves all XML elements with the specified tag name from a parsed XML object.
 *
 * @param {Object} policy - The parsed XML object to search within.
 * @param {string} tagName - The tag name of the XML elements to find.
 * @returns {Array<Object>} An array of all matching XML elements found.
 */
function getAllXMLElements(policy, tagName) {
  const helper = (parent, acc = [ ]) => {
    if (!parent || typeof parent !== 'object') {
      return acc;
    }

    for (const key in parent) {
      if (key.toLowerCase() === tagName.toLowerCase() || key.toLowerCase() === `ns0:${tagName.toLocaleLowerCase()}`) {
        if (Array.isArray(parent[key])) {
          acc.push(...parent[key]);
          continue;
        }
        acc.push(parent[key]);
        continue;
      }

      if (typeof parent[key] === 'object') {
        helper(parent[key], acc);
      }
    }

    return acc;
  }

  return helper(policy);
}


/**
 * Extracts and structures statement information from a given policy XML object.
 *
 * @param {Object} policy - The XML policy object to extract statements from.
 * @returns {Array<Object>} An array of statement objects, each containing:
 *   - data: Array of objects with `ref` (string) and `categories` (string[]).
 *   - purpose: Array of purpose keys (string).
 *   - recipient: Array of recipient keys (string).
 *   - retention: Array of retention keys (string).
 *   - consequence: Array of consequence elements.
 *   - extension: Array of extension elements.
 */
export function getStatements (policy) {
  console.info('%cPolicy info', LOGCOLOR('blue', 'white'), policy);
  const policies = getAllXMLElements(policy, 'policy');
  console.info('%cPolicies found', LOGCOLOR('yellow', 'black'), policies);

  const statements = getAllXMLElements(policy, 'statement');

  return statements.map((statement) => {
    const data = getAllXMLElements(statement, 'data')
      .map((data) => ({
        ref: data['@ref'],
        categories: getAllXMLElements(data, 'categories').flatMap((category) => Object.keys(category)),
      }));
    const purpose = getAllXMLElements(statement, 'purpose')
      .flatMap((purpose) => purpose ? Object.keys(purpose) : [ ]);
    const recipient = getAllXMLElements(statement, 'recipient')
      .flatMap((recipient) => recipient ? Object.keys(recipient) : [ ]);
    const retention = getAllXMLElements(statement, 'retention')
      .flatMap((retention) => retention ? Object.keys(retention) : [ ]);
    const consequence = getAllXMLElements(statement, 'consequence');
    const extension = getAllXMLElements(statement, 'extension');

    return {
      data,
      purpose,
      recipient,
      retention,
      consequence,
      extension,
    };
  });
}
