import {
  attachOpenSettingsListener,
  isRegularURL,
  getHumanReadableDataType,
  getHumanReadableName,
  getPolicy,
  getUserPreferences,
  getViolations,
  LOGCOLOR,
  setURLText,
} from "./common.js";
import {
  LOGEVENTS,
  MESSAGE_TYPES,
  POLICY_PROCESSING_STATUS,
  PREFERENCE_LEVELS,
  STORAGE_KEYS,
} from "./constants.js";
import { getStatements, METADATA_TYPES } from "./p3p.js";
import { setupUUID } from "./user.js";


const BLOCKS = {
  PROCESSING: '.processing',
  NO_POLICY_REQUIRED: '.no-policy-required',
  WARN: '.warn',
  CRITICAL: '.critical',
  NO_POLICY: '.no-policy',
  SUCCESS: '.success',
  DETAILS: '.details',
};


function showBlock(...selectors) {
  return () => {
    Object.values(BLOCKS).forEach(b => {
      document.querySelector(b).style.display = 'none';
    });
    selectors.forEach(selector => {
      document.querySelector(selector).style.display = 'inherit';
    });
  };
}

const showWarning = showBlock(BLOCKS.WARN, BLOCKS.DETAILS);
const showCritical = showBlock(BLOCKS.CRITICAL, BLOCKS.DETAILS);
const showProcessing = showBlock(BLOCKS.PROCESSING);
const showNoPolicyRequired = showBlock(BLOCKS.NO_POLICY_REQUIRED);
const showNoPolicy = showBlock(BLOCKS.NO_POLICY);
const showSuccess = showBlock(BLOCKS.SUCCESS, BLOCKS.DETAILS);
const showSystemMessages = () => {
  const messageContainer = document.querySelector('.system-messages');
  messageContainer.style.display = 'flex';
};
const hideSystemMessages = () => {
  const messageContainer = document.querySelector('.system-messages');
  messageContainer.style.display = 'none';
};

function createStatementItem (statement, violations) {
  if (!statement.data.length) {
    return;
  }

  const statementItem = document.createElement('li');

  const dataGroupList = document.createElement('ul');
  dataGroupList.classList.add('data-group');

  statement.data.forEach(async (data) => {
    const dataTypeListItem = document.createElement('li');
    const dataType = data["ref"];
    const dataTypeHumanReadable = await getHumanReadableDataType(dataType);

    const violatedPreferences = violations.filter((violation) => violation.data === dataType);
    // console.info('%cData type', LOGCOLOR('purple', 'white'), dataType, dataTypeHumanReadable, violatedPreferences);

    const allViolationsUnknown = violatedPreferences.every((violation) => violation.level === -1);

    if (violatedPreferences.length) {
      const preferBlocking = violatedPreferences.some((violation) => violation.level === Object.values(PREFERENCE_LEVELS).indexOf(PREFERENCE_LEVELS.BLOCK));

      dataTypeListItem.classList.add('violation');
      if (allViolationsUnknown) {
        dataTypeListItem.classList.add('unknown');
        dataTypeListItem.classList.remove('block');
      } else if (preferBlocking) {
        dataTypeListItem.classList.add('block');
        dataTypeListItem.classList.remove('warn');
      } else {
        dataTypeListItem.classList.add('warn');
        dataTypeListItem.classList.remove('block');
      }
    } else {
      dataTypeListItem.classList.remove('violation', 'unknown', 'block', 'warn');
    }

    dataTypeListItem.classList.add('data-type');

    dataTypeListItem.title = dataTypeHumanReadable["description"];
    dataTypeListItem.textContent = dataTypeHumanReadable["name"];
    // if (allViolationsUnknown) {
    //   dataTypeListItem.textContent += ' (No preference set)';
    // }
    dataGroupList.appendChild(dataTypeListItem);
  });

  const metaDataTable = document.createElement('table');
  metaDataTable.classList.add('meta-data');

  // For each type of metadata, create a table row displaying it
  for (const type of Object.values(METADATA_TYPES)) {
    const metaDataRow = document.createElement('tr');
    const metaDataNameCell = document.createElement('th');
    metaDataNameCell.textContent = `${type.charAt(0).toUpperCase() + type.slice(1)}: `;
    metaDataRow.appendChild(metaDataNameCell);

    const metaDataCell = document.createElement('td');
    metaDataCell.classList.add(type);

    if (type in statement && Boolean(statement[type]) && Array.isArray(statement[type]) && statement[type].length > 0) {
      const metaDataList = document.createElement('ul');
      statement[type].forEach((item) => {
        const humanReadable = getHumanReadableName(item, type);
        const listItem = document.createElement('li');
        if (!humanReadable) {
          const blockquote = document.createElement('blockquote');
          blockquote.textContent = item;
          listItem.appendChild(blockquote);
          metaDataList.appendChild(listItem);
          return;
        }
        const { name, description } = humanReadable;
        const details = document.createElement('details');
        const summary = document.createElement('summary');
        const paragraph = document.createElement('p');
        summary.textContent = name;
        summary.title = description;
        paragraph.innerText = description;
        details.appendChild(summary);
        details.appendChild(paragraph);
        listItem.appendChild(details);
        metaDataList.appendChild(listItem);
      });
      metaDataCell.appendChild(metaDataList);
    } else {
      const noDataItem = document.createElement('li');
      const span = document.createElement('span');
      span.textContent = 'Not specified';
      noDataItem.appendChild(span);
      metaDataCell.appendChild(noDataItem);
    }
    metaDataRow.appendChild(metaDataCell);
    metaDataTable.appendChild(metaDataRow);
  }

  statementItem.classList.add('statement');
  statementItem.appendChild(dataGroupList);
  statementItem.appendChild(metaDataTable);

  return statementItem;
}


function updatePolicyStatements(policyStatements, violations) {
  const detailsList = document.querySelector('.details ul');
  detailsList.innerHTML = '';

  if (!policyStatements.length || policyStatements.every((statement) => statement.data.length === 0)) {
    const li = document.createElement('li');
    li.textContent = 'No data appears to be processed according to the privacy policy.';
    detailsList.appendChild(li);
    return;
  }

  policyStatements.forEach((statement) => {
    const statementItem = createStatementItem(statement, violations);
    if (statementItem) {
      detailsList.appendChild(statementItem);
    }
  });
}


/**
 * Updates the policy details for a given URL by retrieving the policy,
 * checking for violations against user preferences, and updating the UI
 * accordingly.
 *
 * @async
 * @function updatePolicyDetails
 * @param {string} url - The URL for which to retrieve and process the policy details.
 * @returns {Promise<void>} Resolves when the policy details have been processed and the UI updated.
 */
async function updatePolicyDetails(url) {
  showProcessing();

  getPolicy(url).then(async (policyInfo) => {
    console.log('%cupdatePolicyDetails', LOGCOLOR('blue', 'white'), 'Retrieved policy', url, policyInfo);

    if (policyInfo && policyInfo.status === POLICY_PROCESSING_STATUS.DONE) {
      const  { policy } = policyInfo;

      // Check if policy violates user preferences
      const preferences = await getUserPreferences();
      const statements = getStatements(policy);
      const violations = getViolations(policy, preferences);

      updatePolicyStatements(statements, violations);

      if (violations.length) {
        console.warn('%cPolicy violates user preferences', LOGCOLOR('red', 'white'), url, violations);
        if (violations.some((violation) => violation.level === Object.values(PREFERENCE_LEVELS).indexOf(PREFERENCE_LEVELS.BLOCK))) {
          showCritical();
        } else {
          showWarning();
        }
      } else {
        console.info('%cPolicy does not violate user preferences', LOGCOLOR('green', 'white'), url);
        showSuccess();
      }
    } else {
      showNoPolicy();
    }
  });
}


async function updateSystemMessages () {
  const messages = await chrome.storage.local
    .get(STORAGE_KEYS.BROADCAST_MESSAGES)
    .then(({ [STORAGE_KEYS.BROADCAST_MESSAGES]: messages }) => messages || { });

  const messageContainer = document.querySelector('.system-messages');
  messageContainer.innerHTML = '';


  Object.values(messages).forEach((msg) => {
    if (!msg.acknowledged) {
      const messageElement = document.createElement('div');
      const messageText = document.createElement('p');
      messageText.textContent = msg.message;
      messageElement.appendChild(messageText);
      messageContainer.appendChild(messageElement);
      const ackBtn = document.createElement('button');
      ackBtn.textContent = '✓';
      ackBtn.addEventListener('click', () => {
        msg.acknowledged = new Date().toISOString();
        chrome.storage.local.set({ [STORAGE_KEYS.BROADCAST_MESSAGES]: messages });
        messageElement.remove();
        if (Object.values(messages).every((msg) => msg.acknowledged)) {
          hideSystemMessages();
        }
      });
      messageElement.appendChild(ackBtn);
    }
  });

  if (Object.values(messages).some((msg) => !msg.acknowledged)) {
    showSystemMessages();
  } else {
    hideSystemMessages();
  }
}


async function updatePopup() {
  // Get url of acccently active tab
  const [ tab ] = await chrome.tabs.query({ active: true, currentWindow: true });
  const url = tab.url;

  console.info(`%cLoading popup data...`, LOGCOLOR('blue', 'white'), url);
  setURLText(url);
  if (isRegularURL(url)) {
    updatePolicyDetails(url);
  } else {
    showNoPolicyRequired();
  }

  updateSystemMessages();
}


document.addEventListener("DOMContentLoaded", () => {
  chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.OPEN_POPUP, message: 'Popup opened' });
  setupUUID();
  attachOpenSettingsListener();
  document.querySelector('header .icons .open-about').addEventListener('click', () => chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.NAVIGATE_TO_ABOUT, message: 'About opened' }));

  updatePopup();
});


// Log when the popup is closed
document.addEventListener('beforeunload', () => {
  chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.CLOSE_POPUP, message: 'Popup closed' });
});


// Log when the visibility of the popup changes
document.addEventListener("visibilitychange", () => {
  if (document.visibilityState === "visible") {
    updatePopup();
    chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.OPEN_POPUP, message: 'Popup opened' });
  } else {
    chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.CLOSE_POPUP, message: 'Popup closed' });
  }
});
