import { LOGCOLOR } from "./common.js";
import { LOGEVENTS, MESSAGE_TYPES, PREFERENCE_LEVELS, SLIDER_COLORS, STORAGE_KEYS } from "./constants.js";
import { CATEGORIES, DATA_TYPES_BASE } from "./schema.js";
import { setupUUID } from "./user.js";


document.addEventListener('DOMContentLoaded', () => {
  const preferencesSection = document.querySelector('main section.preferences');

  /**
   * Creates a multi-level selector UI component for preference selection.
   *
   * @param {string} name - The name attribute for the input element.
   * @param {string} labelText - The text to display as the label for the selector.
   * @param {Array} [levels=Object.values(PREFERENCE_LEVELS)] - The available preference levels.
   * @param {boolean} [showLabel=false] - Whether to display the label text.
   * @param {boolean} [showStatus=true] - Whether to display the status label.
   * @returns {HTMLDivElement} The constructed multi-selector DOM element.
   */
  const MultiSelector = (name, labelText, levels = Object.values(PREFERENCE_LEVELS), showLabel = false, showStatus = true) => {
    const multiSelectorElement = document.createElement('div');
    multiSelectorElement.classList.add('multi-selector');
    if (showStatus) {
      multiSelectorElement.classList.add('show-status');
    }
    multiSelectorElement.style.setProperty('--steps', levels.length);

    const label = document.createElement('label');
    label.classList.add('multi-selector-label');
    const nameSpan = document.createElement('span');
    nameSpan.classList.add('multi-selector-name');
    nameSpan.textContent = labelText;
    label.classList.add('multi-selector-label');
    const input = document.createElement('input');

    const switchElement = document.createElement('div');
    switchElement.classList.add('switch-element');

    const statusLabel = document.createElement('span');
    statusLabel.classList.add('status-label');
    statusLabel.textContent = 'Block';

    const switchSpan = document.createElement('span');
    switchSpan.classList.add('switch');

    const anchorWrapper = document.createElement('div');
    anchorWrapper.classList.add('slide-anchor-wrapper');
    Object.values(levels).forEach((level, i) => {
      const anchorSpan = document.createElement('span');
      anchorSpan.classList.add('slide-anchor');
      anchorWrapper.appendChild(anchorSpan);
      anchorSpan.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        setPreference(name, i);
        updateInput(input);
      });
    });
    switchSpan.appendChild(anchorWrapper);

    const sliderSpan = document.createElement('span');
    sliderSpan.classList.add('slider', 'round');
    sliderSpan.style.setProperty('--position', 0);
    switchSpan.appendChild(sliderSpan);

    multiSelectorElement.appendChild(label);
    if (showLabel) {
      label.appendChild(nameSpan);
    }
    label.appendChild(switchElement);

    if (showStatus) {
      switchElement.appendChild(statusLabel);
    }
    switchElement.appendChild(switchSpan);
    label.appendChild(input);
    switchSpan.appendChild(sliderSpan);

    input.type = 'number';
    input.name = name;
    input.min = 0;
    input.max = levels.length;
    input.step = 1;
    input.value = 0;

    label.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      setPreference(name, (parseInt(input.value) + 1) % levels.length);
      updateInput(input);
    });

    input.setMSValue = (value) => {
      if (typeof value !== 'number') {
        value = typeof value === 'string' ? parseInt(value) : 0;
      }

      if (isNaN(value) || value < -1 || value > Object.values(PREFERENCE_LEVELS).length - 1) {
        value = -1;
      }

      input.value = value;

      if (value < 0) {
        sliderSpan.classList.add('indeterminate');
        statusLabel.textContent = '';
        multiSelectorElement.style.setProperty('--slider-color', SLIDER_COLORS.UNDEFINED);
      } else {
        sliderSpan.classList.remove('indeterminate');
        statusLabel.textContent = levels[value] || '';
        multiSelectorElement.style.setProperty('--slider-color', SLIDER_COLORS[levels[value]] || 'inherit');
        sliderSpan.style.setProperty('--position', value);
      }
    };

    return multiSelectorElement;
  }


  /* Preferences settings */

  // Derive the hierarchical grouping of rules
  const RULES = Object.fromEntries(Array.from(Object.keys(CATEGORIES)).map((cat) => [
    cat,
    Object.fromEntries(Object.keys(DATA_TYPES_BASE)
      .filter((rule) => rule.startsWith(cat + '.'))
      .map((key) => [key, DATA_TYPES_BASE[key]]))
  ]));
  // ]).filter(([ category, rules ]) => rules.length > 0));


  // Create UI elements
  Object.keys(RULES).sort().map((category) => [category, RULES[category]]).forEach(([category, rules]) => {
    if (Object.keys(rules).length === 0) {
      return;
    }

    const rulesParentNode = document.createElement((Object.keys(rules).length > 1) ? 'ul' : 'div');
    rulesParentNode.classList.add('preference-rules');

    if (Object.keys(rules).length > 1) {
      const detailsNode = document.createElement('details');
      detailsNode.open = false;
      const summaryNode = document.createElement('summary');
      const summaryHeading = document.createElement('div');
      summaryNode.appendChild(summaryHeading);
      summaryHeading.innerText = CATEGORIES[category];
      const checkbox = MultiSelector(category, CATEGORIES[category], Object.values(PREFERENCE_LEVELS), false, false);
      summaryHeading.appendChild(checkbox);
      detailsNode.appendChild(summaryNode);
      detailsNode.appendChild(rulesParentNode);
      preferencesSection.appendChild(detailsNode);
    } else {
      preferencesSection.appendChild(rulesParentNode);
    }

    Object.entries(rules).forEach(([key, rule]) => {
      const li = document.createElement('li');
      const checkbox = MultiSelector(key, rule.name, Object.values(PREFERENCE_LEVELS), true, true);
      li.appendChild(checkbox);
      rulesParentNode.appendChild(li);
    });
  });


  const preferences = Object.fromEntries(Object.keys(DATA_TYPES_BASE).map((rule) => [rule, Object.values(PREFERENCE_LEVELS).indexOf(PREFERENCE_LEVELS.WARN)]));


  /** Update the input element's value and propagate changes to parent/child elements. */
  const updateInput = (input, propagateToParent=true, propagateToChildren=true) => {
    if (typeof input === 'string') {
      input = document.querySelector(`input[name="${input}"]`);
    }

    const prefix = input ? input.name.split('.').slice(0, -1).join('.') : undefined;

    if (!input) {
      // No input element found, try parent (if applicable) or skip
      if (propagateToParent && prefix) {
        updateInput(prefix, true, false);
      }
      return;
    }

    if (!(input.name in preferences)) {
      // Parent element
      const children = Object.keys(preferences).filter((key) => key.startsWith(input.name + '.') && key !== input.name);

      if (children.length) {
        const refValue = preferences[children[0]];

        // All children have the same value?
        const allEqual = children.every((child) => preferences[child] === refValue);
        const value = allEqual ? refValue : -1;
        if ('setMSValue' in input) {
          input.setMSValue(value);
        } else {
          input.value = value;
        }

        if (propagateToChildren) {
          children.forEach((child) => updateInput(child, false, true));
        }
      }
    } else {
      // child element
      const value = preferences[input.name];

      if ('setMSValue' in input) {
        input.setMSValue(value);
      } else {
        input.value = value;
      }

      if (propagateToParent) {
        if (prefix) {
          updateInput(prefix, true, false);
        }
      }
    }

    return input;
  };


  /** Set the value of a preference in the local object and the extension storage */
  const setPreference = (name, value) => {
    if (typeof value === 'number') {
      value = parseInt(value);
    }

    if (name in preferences) {
      // Set this value directly
      preferences[name] = value;
    } else {
      // Set all children
      Object.keys(preferences)
        .filter((key) => key.startsWith(name + '.') && key !== name)
        .forEach((key) => {
          preferences[key] = value;
        });
    }

    chrome.runtime?.sendMessage?.({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.PREFERENCE_CHANGED, message: 'Preference changed', data: { name, value, preferences } });
    chrome.runtime?.sendMessage?.({ type: MESSAGE_TYPES.PREFERENCES_CHANGED, data: preferences });
    chrome.storage.local.set({
      [STORAGE_KEYS.PREFERENCES]: preferences
    });
  };


  /** Load preferences from storage and init checkboxes */
  chrome.storage.local.get(STORAGE_KEYS.PREFERENCES, (data) => {
    console.info('%cPreferences:', LOGCOLOR('#FF4343', 'white'), data);

    Object.assign(preferences, data.preferences);
    Object.keys(preferences).forEach((key) => {
      const input = document.querySelector(`input[name="${key}"]`);
      if (!input) {
        console.warn('No input found for', key);
        return;
      }
      updateInput(input, true, false);
    });
  });


  const whitelistElement = document.querySelector('main section.whitelist ul');


  /* Whitelisting */
  chrome.storage.local.get(STORAGE_KEYS.WHITELIST, ({ [STORAGE_KEYS.WHITELIST]: whitelist }) => {
    console.info('%cWhitelist:', LOGCOLOR('#FF4343', 'white'), whitelist);

    Object.keys(whitelist).sort().forEach((url) => {
      const listItem = document.createElement('li');
      const textSpan = document.createElement('span');
      textSpan.textContent = url;
      listItem.appendChild(textSpan);

      const rmBtn = document.createElement('button');
      rmBtn.addEventListener('click', () => {
        chrome.runtime?.sendMessage?.({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.REMOVE_WHITELIST, message: 'Removed whitelist exception', data: { url } });
        delete whitelist[url];
        chrome.storage.local.set({ [STORAGE_KEYS.WHITELIST]: whitelist });
        listItem.remove();
      });
      listItem.appendChild(rmBtn);

      whitelistElement.appendChild(listItem);
    });
  });


  /* Tabbing between sections */
  const tabHandles = document.querySelectorAll('nav h2');
  const tabSections = document.querySelectorAll('main section');


  const activateTab = (id) => {
    if (typeof id === 'number') {
      id = tabHandles[id].id;
    }

    if (id === 'preferences') {
      chrome.runtime?.sendMessage?.({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.NAVIGATE_TO_PREFERENCES, message: 'Navigated to preferences' });
    }

    if (id === 'whitelist') {
      chrome.runtime?.sendMessage?.({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.NAVIGATE_TO_WHITELIST, message: 'Navigated to whitelist' });
    }

    sessionStorage.setItem('activeTab', id);

    tabSections.forEach((section) => section.classList.remove('active'));
    tabHandles.forEach((handle) => handle.classList.remove('active'));

    const handle = document.getElementById(id);
    handle.classList.add('active');
    const activeSection = document.querySelector(`main section.${id}`);
    activeSection.classList.add('active');
  };


  tabHandles.forEach((handle) => {
    handle.addEventListener('click', () => {
      activateTab(handle.id);
    });
  });


  const lastActiveTab = sessionStorage.getItem('activeTab');
  activateTab(lastActiveTab || 0);
});


document.addEventListener("DOMContentLoaded", () => {
  setupUUID();
});


chrome.runtime.onMessage.addListener((message) => {
  console.info(message);
  if (message.type === 'p4p:refreshSidepanel') {
    location.reload();
  }
});


document.addEventListener('beforeunload', () => {
  chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.CLOSE_SIDEPANEL, message: 'Sidepanel closed' });
});


document.addEventListener('visibilitychange', () => {
  if (document.visibilityState === 'visible') {
    chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.OPEN_SIDEPANEL, message: 'Sidepanel opened' });
  } else {
    chrome.runtime.sendMessage({ type: MESSAGE_TYPES.LOG, action: LOGEVENTS.CLOSE_SIDEPANEL, message: 'Sidepanel closed' });
  }
});
