import {
  LOGEVENTS,
  MESSAGE_TYPES,
} from './constants.js';


/**
 * Retrieves the UUID from chrome.storage.local, or fetches a new one from the server if missing.
 *
 * @returns {Promise<string|null>} A UUID string, or null if retrieval failed.
 */
export async function getUUID() {
  const {uuid} = await chrome.storage.local.get('uuid');

  if (uuid) {
      return uuid;
  } else {
      console.info('No UUID found, fetching a new one...');
      // Fetch a new UUID from the API
      return fetch('https://p4p.medien.ifi.lmu.de/api/uuid', {
      method: 'POST',
      headers: {
          'Content-Type': 'application/json'
      }
      })
      .then(response => response.json())
      .then(data => {
        chrome.runtime.sendMessage({
          type: MESSAGE_TYPES.LOG,
          action: LOGEVENTS.UUID_SET,
          message: 'UUID fetched from server',
          data,
        });
        const uuid = data.uuid;
        chrome.storage.local.set({ uuid });
      }).catch(error => {
          console.error('Error fetching UUID:', error);
          return null;
      });
  }
};


/**
 * Initializes the UUID display and clipboard copy functionality.
 * Fetches the UUID asynchronously, displays it in any element with class 'uuid',
 * and sets up a click event to copy the UUID to the clipboard.
 *
 * @returns {void}
 */
export function setupUUID () {
  getUUID()
    .then((uuid) => {
      const uuidElements = document.querySelectorAll('.uuid');
      uuidElements.forEach((uuidElement) => {
        uuidElement.textContent = uuid;
        uuidElement.addEventListener('click', () => {
          navigator.clipboard.writeText(uuid).then(() => {
            console.info('UUID copied to clipboard');
            uuidElement.classList.add('copy-success');
            setTimeout(() => {
              uuidElement.classList.remove('copy-success');
            }, 1000);
          }).catch(error => {
            console.error('Error copying UUID to clipboard:', error);
          });
        });
      });
    });
}
